function [a, D] = sppaoli_1008(A, type, is_symmetric, iscorrect)
% 稀疏矩阵A的稀疏泡利分解（仅含type和is_symmetric参数错误处理）
% 输入参数：
%   A           - 待分解的稀疏矩阵
%   type        - 输出类型：'qua'（四进制下标）、'dec'（十进制下标）
%   is_symmetric- 矩阵对称性：true（对称）、false（非对称）
%   iscorrect   - 验证开关：'open'（打开验证）、'off'（关闭验证）
% 输出参数：
%   a           - 泡利分解系数向量 (a0, a1, ...)
%   D           - 泡利矩阵下标矩阵：type='qua'时为四进制，'dec'时为十进制
% 作者：吴锋，李晨，吴炫龙，郭岩松，朱力，杨玉祥
% 2025年9月21日（更新：新增is_symmetric参数错误处理）

%% 1. 第2个参数（type）输入错误处理（保留原逻辑）
if ~strcmp(type, 'qua') && ~strcmp(type, 'dec')
    error('Error: 第2个参数（type）必须为 ''qua'' 或 ''dec''！');
end

%% 2. 第3个参数（is_symmetric）输入错误处理（参照type逻辑实现）
% 核心校验：仅允许布尔值（true/false），非布尔值则抛错
if ~islogical(is_symmetric)
    error('Error: 第3个参数（is_symmetric）必须为布尔值（true（对称） 或 false（非对称））！');
end

%% 3. 矩阵分解分支（使用合法的is_symmetric参数）
if type == 'qua';
    if is_symmetric
        [a, D] = paoli_s(A,0);  % 对称矩阵分解
    else
        [a, D] = paoli_is(A,0); % 非对称矩阵分解
    end
elseif type == 'dec';
    if is_symmetric
        [a, D] = paoli_s(A,1);  % 对称矩阵分解
    else
        [a, D] = paoli_is(A,1); % 非对称矩阵分解
    end
else
    error('Error, the second input parameter must be dec or qua!' );
end

%% 4. 原验证逻辑（保持不变，不新增其他参数校验）
if nargin >= 4
    if strcmp(iscorrect, 'open')
        % 泡利矩阵库初始化
        PL = cell(4, 1);
        PL{1} = eye(2);
        PL{2} = [1 0; 0 -1];
        PL{3} = [0 1; 1 0];
        PL{4} = [0 -sqrt(-1); sqrt(-1) 0];
        
        N = size(A, 1);
        n = log2(N);
        Ki = sparse(N, N);
        Nd = length(a);
        
        % 重构矩阵并计算误差
        for i = 1:Nd
            if strcmp(type, 'qua')
                q = D(:, i) + 1;
            elseif strcmp(type, 'dec')
                q = dec2quat_vector(double(D(i)), n) + 1;
            end
            Gi = sparse(PL{q(1)});
            for k = 2:n
                Gi = kron(Gi, PL{q(k)});
            end
            Ki = Ki + Gi * a(i);
        end
        
        er = norm(A - Ki, 1);
        if er <= 1e-12
            disp(['泡利分解正确，误差（1范数）：', num2str(er)]);
        else
            disp(['泡利分解错误，误差（1范数）：', num2str(er)]);
        end
    elseif strcmp(iscorrect, 'off')
        disp('泡利分解验证部分已经关闭');
    else
        error('Error: 第4个参数（iscorrect）必须为 ''open'' 或 ''off''！');
    end
end